// Copyright 2013 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

goog.require('axs.AuditRules');
goog.require('axs.browserUtils');
goog.require('axs.constants.Severity');

/**
 * The purpose of each link should be clear from the link text.
 */
axs.AuditRules.addRule({
    name: 'linkWithUnclearPurpose',
    heading: 'The purpose of each link should be clear from the link text',
    url: 'https://github.com/GoogleChrome/accessibility-developer-tools/wiki/Audit-Rules#ax_text_04',
    severity: axs.constants.Severity.WARNING,
    /**
     * @param {Element} element
     * @return {boolean}
     */
    relevantElementMatcher: function(element, flags) {
        return axs.browserUtils.matchSelector(element, 'a[href]') && !flags.hidden;
    },
    /**
     * @param {Element} anchor
     * @param {Object=} opt_config
     * @return {boolean}
     */
    test: function(anchor, flags, opt_config) {
        var config = opt_config || {};
        var blacklistPhrases = config['blacklistPhrases'] || [];
        var whitespaceRE = /\s+/;
        for (var i = 0; i < blacklistPhrases.length; i++) {
            // Match the blacklist phrase, case insensitively, as the whole string (allowing for
            // punctuation at the end).
            // For example, a blacklist phrase of "click here" will match "Click here." and
            // "click here..." but not "Click here to learn more about trout fishing".
            var phraseREString =
                '^\\s*' + blacklistPhrases[i].trim().replace(whitespaceRE, '\\s*') + '\s*[^a-z]$';
            var phraseRE = new RegExp(phraseREString, 'i');
            if (phraseRE.test(anchor.textContent))
                return true;
        }

        // Remove punctuation from phrase, then strip out all stopwords. Fail if remaining text is
        // all whitespace.
        var stopwords = config['stopwords'] ||
            ['click', 'tap', 'go', 'here', 'learn', 'more', 'this', 'page', 'link', 'about'];
        var filteredText = axs.properties.findTextAlternatives(anchor, {});
        if (filteredText === null || filteredText.trim() === '')
            return true;
        filteredText = filteredText.replace(/[^a-zA-Z ]/g, '');
        for (var i = 0; i < stopwords.length; i++) {
            var stopwordRE = new RegExp('\\b' + stopwords[i] + '\\b', 'ig');
            filteredText = filteredText.replace(stopwordRE, '');
            if (filteredText.trim() == '')
                return true;
        }
        return false;
    },
    code: 'AX_TEXT_04'
});
